﻿/*!
 * Author: (Passion-Creator)
 * License: MIT License
*/
document.addEventListener('DOMContentLoaded', function () {

    // 1. ヘッダーのスクロールエフェクト
    // ページをスクロールした際にヘッダーのデザインを変更します。
    const header = document.querySelector('header');
    window.addEventListener('scroll', () => {
        if (window.scrollY > 50) {
            header.classList.add('scrolled');
        } else {
            header.classList.remove('scrolled');
        }
    });

    // 2. スクロールに応じたフェードインアニメーション
    // 各セクションが画面に入ってきたら、フワッと表示させます。
    const fadeInElements = document.querySelectorAll('.fade-in');
    const fadeInObserver = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('is-visible');
                // 一度表示されたら、以降は監視を停止してパフォーマンスを向上させます。
                fadeInObserver.unobserve(entry.target);
            }
        });
    }, {
        threshold: 0.1 // 要素が10%見えたらアニメーションを開始します。
    });

    fadeInElements.forEach(el => {
        fadeInObserver.observe(el);
    });

    // 3. FAQのアコーディオン機能
    // 質問をクリックすると、回答が開閉します。
    const faqItems = document.querySelectorAll('.faq-item');
    faqItems.forEach(item => {
        const question = item.querySelector('.faq-question');
        const answer = item.querySelector('.faq-answer');

        question.addEventListener('click', () => {
            // 他の開いている項目があれば、それを閉じます。
            faqItems.forEach(otherItem => {
                if (otherItem !== item && otherItem.classList.contains('active')) {
                    otherItem.classList.remove('active');
                    otherItem.querySelector('.faq-answer').style.maxHeight = null;
                }
            });

            // クリックされた項目を開閉します。
            item.classList.toggle('active');
            if (item.classList.contains('active')) {
                // `scrollHeight` を使って、コンテンツに応じた高さを取得し設定します。
                answer.style.maxHeight = answer.scrollHeight + 'px';
            } else {
                answer.style.maxHeight = null;
            }
        });
    });
    
    // 4. 数字のカウントアップアニメーション
    // 「数字で見る私たち」セクションが表示されたら、数字が増えていくアニメーションを実行します。
    const numbersSection = document.querySelector('#numbers');

    const animateCounters = () => {
        const counters = numbersSection.querySelectorAll('.number');
        const duration = 2000; // アニメーションにかかる時間（2秒）

        counters.forEach(counter => {
            const target = +counter.dataset.target; // HTMLのdata-target属性から最終的な数値を取得します。
            
            // "年"や"%"などの単位（<span>タグ）を保持します。
            const unit = counter.querySelector('span') ? counter.querySelector('span').outerHTML : '';
            // カウンター本体のテキスト部分だけを更新対象にします。
            const numberElement = counter.childNodes[0];

            let startTimestamp = null;
            const step = (timestamp) => {
                if (!startTimestamp) startTimestamp = timestamp;
                const progress = Math.min((timestamp - startTimestamp) / duration, 1);
                const currentValue = Math.floor(progress * target);
                
                // 画面上の数値を更新します。
                numberElement.textContent = currentValue;
                
                if (progress < 1) {
                    // アニメーションが完了するまで、次のフレームを要求します。
                    window.requestAnimationFrame(step);
                } else {
                    // 最後に、最終的な数値を正確に表示します。
                    numberElement.textContent = target;
                }
            };
            // アニメーションを開始します。
            window.requestAnimationFrame(step);
        });
    };

    // 数字セクションが画面に表示されたらアニメーションを開始するための監視設定
    const numberObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                animateCounters();
                observer.unobserve(entry.target); // アニメーションは1回だけ実行します。
            }
        });
    }, {
        threshold: 0.5 // セクションが50%見えたら開始します。
    });

    // #numbersセクションの監視を開始します。
    if (numbersSection) {
        numberObserver.observe(numbersSection);
    }

});